/*
 *  $Id: surface.c 28514 2025-09-04 17:58:57Z yeti-dn $
 *  Copyright (C) 2025 David Nečas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "tests/testlibgwy.h"

static void
assert_surface_content(GwySurface *surface, const GwyXYZ *ref_data, guint ref_n, gdouble eps)
{
    g_assert_true(GWY_IS_SURFACE(surface));
    guint n = gwy_surface_get_npoints(surface);
    g_assert_cmpuint(n, ==, ref_n);
    const GwyXYZ *data = gwy_surface_get_data_const(surface);
    g_assert_nonnull(data);

    if (eps > 0.0) {
        for (guint i = 0; i < ref_n; i++) {
            g_assert_cmpfloat_with_epsilon(data[i].x, ref_data[i].x, eps);
            g_assert_cmpfloat_with_epsilon(data[i].y, ref_data[i].y, eps);
            g_assert_cmpfloat_with_epsilon(data[i].z, ref_data[i].z, eps);
        }
    }
    else {
        for (guint i = 0; i < ref_n; i++) {
            g_assert_cmpfloat(data[i].x, ==, ref_data[i].x);
            g_assert_cmpfloat(data[i].y, ==, ref_data[i].y);
            g_assert_cmpfloat(data[i].z, ==, ref_data[i].z);
        }
    }
}

static void
surface_assert_equal(GObject *object, GObject *reference)
{
    g_assert_true(GWY_IS_SURFACE(object));
    g_assert_true(GWY_IS_SURFACE(reference));

    GwySurface *surface = GWY_SURFACE(object), *surface_ref = GWY_SURFACE(reference);
    g_assert_cmpint(gwy_surface_get_npoints(surface), ==, gwy_surface_get_npoints(surface_ref));
    g_assert_true(gwy_unit_equal(gwy_surface_get_unit_xy(surface), gwy_surface_get_unit_xy(surface_ref)));
    g_assert_true(gwy_unit_equal(gwy_surface_get_unit_z(surface), gwy_surface_get_unit_z(surface_ref)));
    assert_surface_content(surface, gwy_surface_get_data(surface_ref), gwy_surface_get_npoints(surface_ref), 0.0);
}

void
test_surface_basic(void)
{
    const GwyXYZ xyz[5] = {
        { 1, 2, 3 },
        { G_PI, G_PI, G_PI },
        { -1.3, 0.0, 330.0 },
        { 10.5, 11.5, -4.25 },
        { 1e-6, 1e-7, 1e-9 },
    };

    GwySurface *surface = gwy_surface_new_from_data(xyz, 5);
    assert_surface_content(surface, xyz, 5, 0.0);

    g_assert_finalize_object(surface);
}

void
test_surface_data_changed(void)
{
    GwySurface *surface = gwy_surface_new();
    guint item_changed = 0;
    g_signal_connect_swapped(surface, "data-changed", G_CALLBACK(record_signal), &item_changed);
    gwy_surface_data_changed(surface);
    g_assert_cmpuint(item_changed, ==, 1);
    gwy_surface_data_changed(surface);
    g_assert_cmpuint(item_changed, ==, 2);
    g_assert_finalize_object(surface);
}

static GwySurface*
create_surface_for_serialisation(void)
{
    const GwyXYZ xyz[5] = {
        { 1, 2, 3 },
        { G_PI, G_PI, G_PI },
        { -1.3, 0.0, 330.0 },
        { 10.5, 11.5, -4.25 },
        { 1e-6, 1e-7, 1e-9 },
    };

    GwySurface *surface = gwy_surface_new_sized(5);
    GwyXYZ *data = gwy_surface_get_data(surface);
    for (guint i = 0; i < 5; i++)
        data[i] = xyz[i];

    return surface;
}

static void
set_surface_units_for_serialisation(GwySurface *surface)
{
    gwy_unit_set_from_string(gwy_surface_get_unit_xy(surface), "m");
    gwy_unit_set_from_string(gwy_surface_get_unit_z(surface), "V");
}

void
test_surface_serialization(void)
{
    GwySurface *surface = create_surface_for_serialisation();
    serialize_object_and_back(G_OBJECT(surface), surface_assert_equal, FALSE, NULL);

    set_surface_units_for_serialisation(surface);
    serialize_object_and_back(G_OBJECT(surface), surface_assert_equal, FALSE, NULL);

    g_assert_finalize_object(surface);
}

void
test_surface_copy(void)
{
    GwySurface *surface = create_surface_for_serialisation();
    serializable_test_copy(GWY_SERIALIZABLE(surface), surface_assert_equal);

    set_surface_units_for_serialisation(surface);
    serializable_test_copy(GWY_SERIALIZABLE(surface), surface_assert_equal);

    g_assert_finalize_object(surface);
}

void
test_surface_assign(void)
{
    GwySurface *surface = create_surface_for_serialisation();
    serializable_test_assign(GWY_SERIALIZABLE(surface), NULL, surface_assert_equal);

    set_surface_units_for_serialisation(surface);
    serializable_test_assign(GWY_SERIALIZABLE(surface), NULL, surface_assert_equal);

    const GwyXYZ xyz[1] = { { 7, 8, 9 } };
    GwySurface *another = gwy_surface_new_from_data(xyz, 1);
    gwy_unit_set_from_string(gwy_surface_get_unit_xy(another), "A");
    gwy_unit_set_from_string(gwy_surface_get_unit_z(another), "kg");
    serializable_test_assign(GWY_SERIALIZABLE(surface), GWY_SERIALIZABLE(another), surface_assert_equal);
    g_assert_finalize_object(another);

    g_assert_finalize_object(surface);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
